package furny.states;

import furny.ga.FurnLayoutIndividual;
import furny.ga.util.FurnLayoutIOUtil;
import ga.core.evaluation.IInteractiveFitnessEvaluator;
import ga.view.appstate.menu.IMenuListenerParent;
import ga.view.appstate.menu.MenuListener;
import ga.view.streaming.showroom.BoxShowRoom.LightingType;
import ga.view.streaming.showroom.ShowRoom;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JColorChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;

import com.jme3.math.ColorRGBA;

import de.lessvoid.nifty.screen.DefaultScreenController;
import de.lessvoid.nifty.tools.Color;

/**
 * This is a controller that allows communication with the nifty gui.
 * 
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class InitializerUIController extends DefaultScreenController implements
    IMenuListenerParent {

  private final IInteractiveFitnessEvaluator<FurnLayoutIndividual> evaluator;
  private final InitializerPreState state;

  private final List<String> floorTextures;

  private MenuListener menuListener;

  private ColorRGBA wallColor = ShowRoom.DEFAULT_COLOR;

  /**
   * Instantiates a new initializer ui controller.
   * 
   * @param evaluator
   *          the evaluator
   * @param state
   *          the state
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public InitializerUIController(
      final IInteractiveFitnessEvaluator<FurnLayoutIndividual> evaluator,
      final InitializerPreState state) {
    this.evaluator = evaluator;
    this.state = state;

    floorTextures = new ArrayList<String>();
    addTexture("Wood_Floor_Dark_1.j3m");
    addTexture("Wood_Floor_Light_1.j3m");
    addTexture("Wood_Floor_Parquet_1.j3m");
    addTexture("Wood_Lumber_ButtJoined_1.j3m");
    addTexture("Tiles_Checker.j3m");
    addTexture("Tiles_White.j3m");
    addTexture("Carpet_Berber_Pattern_Gray_1.j3m");
    addTexture("Carpet_Loop_Pattern_1.j3m");
    addTexture("Carpet_Plush_Charcoal_1.j3m");
  }

  /**
   * Adds a texture.
   * 
   * @param file
   *          the filename
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void addTexture(final String file) {
    floorTextures.add("furny/textures/" + file);
  }

  /**
   * Sets the wall color.
   * 
   * @param wallColor
   *          the new wall color
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setWallColor(final ColorRGBA wallColor) {
    this.wallColor = wallColor;
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void onTerminate() {
    state.saveSettings();

    if (menuListener != null) {
      menuListener.nextState();
    }
  }

  @Override
  public void setMenuListener(final MenuListener menuListener) {
    this.menuListener = menuListener;
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void onFloorTextureChange(final String num) {
    final int i = Integer.parseInt(num);

    state.setFloorTexture(floorTextures.get(i));
  }

  /**
   * Gets the room size as string.
   * 
   * @return the room size string
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public String getRoomSizeString() {
    return state.getRoomSizeString();
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void onPopPercentageChange() {
    SwingUtilities.invokeLater(new Runnable() {
      @Override
      public void run() {
        final JPanel panel = new JPanel(new GridBagLayout());
        final GridBagConstraints constraints = new GridBagConstraints();
        constraints.insets = new Insets(5, 5, 5, 5);
        constraints.gridx = 0;
        constraints.gridy = 0;
        constraints.fill = GridBagConstraints.BOTH;
        constraints.gridwidth = 1;
        constraints.gridheight = 1;
        constraints.weightx = 0.0d;
        constraints.weighty = 0.0d;

        final int oldPerc = state.getPopPercentage();

        panel
            .add(new JLabel(
                "Initialize % of the Population with this Individual"),
                constraints);

        constraints.gridx++;
        final SpinnerNumberModel model = new SpinnerNumberModel(oldPerc, 1,
            100, 1);
        final JSpinner percentageSpinner = new JSpinner(model);
        panel.add(percentageSpinner, constraints);

        final int n = JOptionPane.showConfirmDialog(null, panel,
            "Select Percentage of this Individual",
            JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);

        if (n == JOptionPane.OK_OPTION) {
          final int newPerc = (Integer) percentageSpinner.getValue();

          if (newPerc != oldPerc) {
            state.setPopPercentage(newPerc);
          }
        }
      }
    });
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void onRoomSizeChange() {
    SwingUtilities.invokeLater(new Runnable() {
      @Override
      public void run() {
        final JPanel panel = new JPanel(new GridBagLayout());
        final GridBagConstraints constraints = new GridBagConstraints();
        constraints.insets = new Insets(5, 5, 5, 5);
        constraints.gridx = 0;
        constraints.gridy = 0;
        constraints.fill = GridBagConstraints.BOTH;
        constraints.gridwidth = 1;
        constraints.gridheight = 1;
        constraints.weightx = 0.0d;
        constraints.weighty = 0.0d;

        final int oldWidth = state.getRoomWidth();
        final int oldLength = state.getRoomLength();
        final int oldHeight = state.getRoomHeight();

        panel.add(new JLabel("Room Width (cm)"), constraints);

        constraints.gridx++;
        SpinnerNumberModel model = new SpinnerNumberModel(oldWidth, 50, 5000, 5);
        final JSpinner widthSpinner = new JSpinner(model);
        panel.add(widthSpinner, constraints);

        constraints.gridx++;
        panel.add(new JLabel("Room Length (cm)"), constraints);

        constraints.gridx++;
        model = new SpinnerNumberModel(oldLength, 50, 5000, 5);
        final JSpinner lengthSpinner = new JSpinner(model);
        panel.add(lengthSpinner, constraints);

        constraints.gridx++;
        panel.add(new JLabel("Room Height (cm)"), constraints);

        constraints.gridx++;
        model = new SpinnerNumberModel(oldHeight, 180, 350, 5);
        final JSpinner heightSpinner = new JSpinner(model);
        panel.add(heightSpinner, constraints);

        final int n = JOptionPane.showConfirmDialog(null, panel,
            "Select Room Size", JOptionPane.OK_CANCEL_OPTION,
            JOptionPane.PLAIN_MESSAGE);

        if (n == JOptionPane.OK_OPTION) {
          final int newWidth = (Integer) widthSpinner.getValue();
          final int newLength = (Integer) lengthSpinner.getValue();
          final int newHeight = (Integer) heightSpinner.getValue();

          if (newWidth != oldWidth || newLength != oldLength
              || newHeight != oldHeight) {
            state.setRoomSize(newWidth, newLength, newHeight);
          }
        }
      }
    });
  }

  /**
   * Gets the pop percentage string.
   * 
   * @return the pop percentage string
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public String getPopPercentageString() {
    return state.getPopPercentageString();
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @param colorString
   *          The new color as string.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void onWallColorChange(final String colorString) {

    final Color niftyC = new Color(colorString);

    state.setWallColor(new ColorRGBA(niftyC.getRed(), niftyC.getGreen(), niftyC
        .getBlue(), niftyC.getAlpha()));
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void onLoadIndividual() {
    SwingUtilities.invokeLater(new Runnable() {
      @Override
      public void run() {
        final FurnLayoutIndividual ind = FurnLayoutIOUtil.loadGenotype(null,
            new FurnLayoutIndividual(evaluator.getAlgorithm().getContext()));
        if (ind != null) {
          state.setIndividual(ind);
        }
      }
    });
  }

  /**
   * Gets the individual.
   * 
   * @return the individual
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public FurnLayoutIndividual getIndividual() {
    return state.getIndividual();
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void onResetIndividual() {
    state.setIndividual(null);
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @param lightingType
   *          The new type of lighting.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void onLightingChange(final String lightingType) {
    try {
      final LightingType type = LightingType.valueOf(lightingType);

      state.setLightingType(type);
    } catch (final Exception e) {
      // ignore
    }
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void onWallColorSelect() {
    SwingUtilities.invokeLater(new Runnable() {
      @Override
      public void run() {
        final java.awt.Color oldAwtC = new java.awt.Color(wallColor.getRed(),
            wallColor.getGreen(), wallColor.getBlue());

        final java.awt.Color awtC = JColorChooser.showDialog(null,
            "Select Wall Color", oldAwtC);

        if (awtC != null && !oldAwtC.equals(awtC)) {
          wallColor = new ColorRGBA(awtC.getRed() / 255f,
              awtC.getGreen() / 255f, awtC.getBlue() / 255f,
              awtC.getAlpha() / 255f);
          state.setWallColor(wallColor);
        }
      }
    });

  }

  /**
   * Gets the wall color as string.
   * 
   * @return the color string
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public String getColorString() {
    return getColor().getColorString();
  }

  /**
   * Gets the wall color.
   * 
   * @return the color
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public Color getColor() {
    return new Color(wallColor.getRed(), wallColor.getGreen(),
        wallColor.getBlue(), wallColor.getAlpha());
  }

  /**
   * Gets the image file at index i.
   * 
   * @param i
   *          the index of the image file.
   * @return the image file
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public String getImageFile(final int i) {
    return floorTextures.get(i).replace(".j3m", ".jpg");
  }
}
